//------------------------
// PARAMETERS

float4x4 gWorldViewProjMatrix
<
   string paramClass="intrinsic";
   string paramName="worldViewProj";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4x4 gWorldViewMatrix
<
   string paramClass="intrinsic";
   string paramName="worldView";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4x4 gWorldMatrix
<
   string paramClass="intrinsic";
   string paramName="world";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4 gModelSpaceCameraPos
<
   string paramClass="intrinsic";
   string paramName="modelSpaceCameraPos";
> = float4(0,0,0,1);

float4x4 gWorldShadowMatrix
<
   string paramClass="intrinsic";
   string paramName="shadowMatrix";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4 gShadowXBiasYRange
<
   string paramClass="intrinsic";
   string paramName="shadow_XRange_YBias";
> = float4(0, 0, 0, 0);

float4x4 gLightProjectedMatrix
<
   string paramClass="intrinsic";
   string paramName="lightProjectedMatrix";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4 gSunDir
<
   string paramClass="intrinsic";
   string paramName="sunDirection";
> = float4(0, 0, 1, 0);

float4 gPreMultSun
<
   string paramClass="intrinsic";
   string paramName="PreMultSun";
> = float4(1,1,1,1);

float4 gSunColorLinear
<
  string paramClass="intrinsic";
  string paramName="sunColor";
  string paramType="color";
  bool linearColor=true;
> = float4(1, 1, 1, 1);

float gSunIntensity
<
  string paramClass="intrinsic";
  string paramName="sunIntensity";
  string paramType="float";
> = 1.0;

float4 gAmbientColorLinear
<
  string paramClass="intrinsic";
  string paramName="ambientColor";
  string paramType="color";
  bool linearColor=true;
> = float4(0, 0, 0, 0);

float gFogDensity
<
   string paramClass="intrinsic";
   string paramName="FogDensity";
   string paramType="float";
> = 0.0f;

float gFogStart
<
   string paramClass="intrinsic";
   string paramName="FogStart";
   string paramType="float";
> = 0.0;

float gPlanarFogDensity
<
  string paramClass="intrinsic";
  string paramName="PlanarFogDensity";
  string paramType="float";
> = 0.0;

float4 gPlanarFogEquation
<
  string paramClass="intrinsic";
  string paramName="PlanarFogEquation";
> = float4(0,1,0,0);

float4 gHemiAxis
<
   string paramClass="intrinsic";
   string paramName="HemiAxis";
> = float4(0,1,0,0);

float4 gHemiTopColor
<
   string paramClass="intrinsic";
   string paramName="HemiTopColor";
   string paramType="color";
> = float4(1,0,0,1);

float4 gHemiBottomColor
<
   string paramClass="intrinsic";
   string paramName="HemiBottomColor";
   string paramType="color";
> = float4(1,0,0,1);

float gHemiIntensity
<
   string paramClass="intrinsic";
   string paramName="HemiIntensity";
   string paramType="float";
> = 1.0;

float4 gFlatColor
<
  string paramClass="intrinsic";
  string paramName="flatColor";
  string paramType="color";
> = float4(0, 0, 0, 0);

float4 gFlatColorLinear
<
  string paramClass="intrinsic";
  string paramName="flatColor";
  string paramType="color";
  bool linearColor=true;
> = float4(0, 0, 0, 0);

float4 gTintColorLinear
<
  string paramClass="intrinsic";
  string paramName="flatColor";
  string paramType="color";
  bool linearColor=true;
> = float4(0, 0, 0, 0);

float gRGBScale
<
   string paramClass="intrinsic";
   string paramName="RGBScale";
   string paramType="float";
> = 1.0f;

float4 gTreeColorVariation
<
   string paramClass="param";
   string paramName="treeVariationColor";
   string paramType="color";
> = float4(0,0,0,0);

float4 gBlendColor
<
  string paramClass="param";
  string paramName="blendColor";
  string paramType="color";
> = float4(0,0,0,0);

float4 gPixelXFormColorLinear
<
   string paramClass="param";
   string paramName="pixelxformColor";
   string paramType="color";
   bool linearColor=true;
> = float4(1, 1, 1, 1);

float4 gPixelXFormColor
<
   string paramClass="param";
   string paramName="pixelxformColor";
   string paramType="color";
> = float4(1, 1, 1, 1);

float gBumpScale
<
   string paramClass="intrinsic";
   string paramName="BumpScale";
   string paramType="float";
> = 1.0f;

float gTerrainBumpScale
<
   string paramClass="intrinsic";
   string paramName="TerrainBumpScale";
   string paramType="float";
> = 0.0f;

float gTerrainMetalness
<
   string paramClass="intrinsic";
   string paramName="TerrainMetalness";
   string paramType="float";
> = 0.25f;

float gSceneMetalness
<
   string paramClass="intrinsic";
   string paramName="SceneMetalness";
   string paramType="float";
> = 0.25f;

float gPreMultSunTerrainSpec
<
   string paramClass="intrinsic";
   string paramName="PreMultSunTerrainSpec";
   string paramType="float";
> = 1.0f;

float gPreMultSunSceneSpec
<
   string paramClass="intrinsic";
   string paramName="PreMultSunSceneSpec";
   string paramType="float";
> = 1.0f;

float gTerrainSpecularPower
<
   string paramClass="intrinsic";
   string paramName="TerrainSpecularPower";
   string paramType="float";
> = 0.0f;

float gSceneSpecularPower
<
   string paramClass="intrinsic";
   string paramName="SceneSpecularPower";
   string paramType="float";
> = 16.0f;

float4 gPlanarFogColor
<
string paramClass="intrinsic";
string paramName="PlanarFogColor";
string paramType="color";
> = float4(0,0,0,1);

float4 gFogColor
<
   string paramClass="intrinsic";
   string paramName="FogColor";
   string paramType="color";
> = float4(0,0,0,1);

float4 gSparkleOffset
<
   string paramClass="param";
   string paramName="sparkleOffset";
> = float4(0,0,0,0);

float4 gShadowColor
<
   string paramClass="intrinsic";
   string paramName="shadowColor";
   string paramType="color";
> = float4(0,0,0,0);

float4 gShadowPCFOffset0
<
   string paramClass="intrinsic";
   string paramName="ShadowPCFOffset0";
> = float4(0.5,0.5,0.5,0.5);

float4 gShadowPCFOffset1
<
   string paramClass="intrinsic";
   string paramName="ShadowPCFOffset1";
> = float4(0,0,0,0);

float4 gShadowPCFOffset2
<
   string paramClass="intrinsic";
   string paramName="ShadowPCFOffset2";
> = float4(0,0,0,0);

float4 gTreeWindHack
<
  string paramClass="param";
  string paramName="treewindhack";
> = float4(0,0,0,0);

sampler gSampler0 : register(s0);
sampler gSampler1 : register(s1);
sampler gSampler2 : register(s2);
sampler gSampler3 : register(s3);
sampler gSampler4 : register(s4);
sampler gSampler7 : register(s7);

// END PARAMETERS


//------------------------
// SUBROUTINES

vec4 SRGBToLinear(vec4 c)
{
   vec4 ret = c;
   ret.rgb *= ret.rgb;
   return ret;
}

float ComputeSquaredExponentialFogDensity(float fogDistance)
{
   float  squaredExponent = fogDistance * fogDistance;      
   float  finalFogDensity = clamp( exp(-squaredExponent), 0.0, 1.0);
   return finalFogDensity;   
}

#ifdef SHADOWING_NV

vec3 getShadowValueNV( sampler2DShadow tx, vec4 texCoord0, vec4 texCoord1, vec4 texCoord2, float shadowColor )
{
	vec3 shadowVals;
	vec4 averageVec = vec4(0.25);
	
	vec4 shadowVec = shadow2DProj(tx, texCoord0);
	shadowVals.x = dot( shadowVec, averageVec );
	
	shadowVec = shadow2DProj(tx, texCoord1);
	shadowVals.y = dot( shadowVec, averageVec );
	
	shadowVec = shadow2DProj(tx, texCoord2);
	shadowVals.z = dot( shadowVec, averageVec );
	
	float shadowScalar = dot( shadowVals, vec3(0.33333) );

	shadowScalar = mix(shadowColor, 1.0, shadowScalar);
	return( vec3(shadowScalar) );
}

#else

#ifdef SHADOWMAP_SIZE
const vec2 shadow_scale = vec2( float(SHADOWMAP_SIZE), float(SHADOWMAP_SIZE) );
const vec2 shadow_inv_scale = vec2( float(INV_SHADOWMAP_SIZE), float(INV_SHADOWMAP_SIZE) );
#endif

vec3 getShadowingValue(sampler2D tx, vec4 shadowmapTexCoord, float lightSpacePixelDepth, float shadowColor )
{
	// Invert y-coord from shadow map for GL.
	shadowmapTexCoord.y = 1.0 - shadowmapTexCoord.y;

#if defined(_SHADOWS_FILTER_2x2_BILINEAR_)

	shadowmapTexCoord.xy *= shadow_scale;
	vec2 fracCoords = fract( shadowmapTexCoord.xy );
	shadowmapTexCoord.xy -= fracCoords;
	shadowmapTexCoord.xy *= shadow_inv_scale;

	vec4 shadowDepth;
	shadowDepth.x = texture2D( tx, shadowmapTexCoord.xy ).x;
	shadowDepth.y = texture2D( tx, vec2(shadowmapTexCoord.x, shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET) ).x;
	shadowDepth.z = texture2D( tx, vec2(shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET, shadowmapTexCoord.y) ).x;
	shadowDepth.w = texture2D( tx, vec2(shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET, shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET) ).x;

	vec4 shadows = step( lightSpacePixelDepth, shadowDepth  );

	vec2 temp = mix(shadows.xy, shadows.zw, fracCoords.x);
	float sv = mix(temp.x, temp.y, fracCoords.y);

#elif defined(_SHADOWS_FILTER_3x3_BILINEAR_)
	
	shadowmapTexCoord.xy *= shadow_scale;
	vec2 fracCoords = fract( shadowmapTexCoord.xy );
	shadowmapTexCoord.xy -= fracCoords;
	shadowmapTexCoord.xy *= shadow_inv_scale;

	vec3 leftCol, midCol, rightCol;
	//-- Sampling
	// Column 1
	leftCol.x = texture2D(tx, vec2( shadowmapTexCoord.x - SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y - SHADOWMAP_TEXOFFSET)   ).x;
	leftCol.y = texture2D(tx, vec2( shadowmapTexCoord.x - SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y)   ).x;
	leftCol.z = texture2D(tx, vec2( shadowmapTexCoord.x - SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET)   ).x;
	// Column 2
	midCol.x = texture2D(tx, vec2( shadowmapTexCoord.x,
									shadowmapTexCoord.y - SHADOWMAP_TEXOFFSET)   ).x;
	midCol.y = texture2D(tx, vec2( shadowmapTexCoord.x,
									shadowmapTexCoord.y)   ).x;
	midCol.z = texture2D(tx, vec2( shadowmapTexCoord.x,
									shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET)   ).x;
	// Column 3
	rightCol.x = texture2D(tx, vec2( shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y - SHADOWMAP_TEXOFFSET)   ).x;
	rightCol.y = texture2D(tx, vec2( shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y)   ).x;
	rightCol.z = texture2D(tx, vec2( shadowmapTexCoord.x + SHADOWMAP_TEXOFFSET,
									shadowmapTexCoord.y + SHADOWMAP_TEXOFFSET)   ).x;

	// Comparing
	vec3 pixDepth = vec3(lightSpacePixelDepth);
	leftCol  = step( pixDepth, leftCol );	// 0 or 1, depending on comparison
	midCol   = step( pixDepth, midCol );
	rightCol = step( pixDepth, rightCol );

	// Filtering
	vec3 lerpedX1 = mix(leftCol, midCol, fracCoords.x);
	vec3 lerpedX2 = mix(midCol, rightCol, fracCoords.x);

	vec4 interpolateY;
	interpolateY.x = mix(lerpedX1.x, lerpedX1.y, fracCoords.y);
	interpolateY.y = mix(lerpedX1.y, lerpedX1.z, fracCoords.y);
	interpolateY.z = mix(lerpedX2.x, lerpedX2.y, fracCoords.y);
	interpolateY.w = mix(lerpedX2.y, lerpedX2.z, fracCoords.y);

	float sv = dot(interpolateY, vec4(0.25));

#else

	// No filtering
	vec4 mapDepth = texture2D(tx, shadowmapTexCoord.st);
	float sv = (lightSpacePixelDepth >= mapDepth) ? 0.0 : 1.0;

#endif

	sv = mix(shadowColor, 1.0, sv);
	return( vec3(sv) );
}

#endif

// END SUBROUTINES


//------------------------
// FUNCTION_VS DefaultVS

uniform mat4  gWorldViewProjMatrix;
uniform mat4  gWorldViewMatrix;
uniform vec4  gSunDir;
uniform vec4  gPreMultSun;
uniform vec4  gHemiAxis;
uniform vec4  gHemiTopColor;
uniform vec4  gHemiBottomColor;
uniform float gHemiIntensity;
uniform float gFogDensity;
uniform float gFogStart;
uniform float gPlanarFogDensity;
uniform vec4  gPlanarFogEquation;

varying vec4 DirAccum;
varying vec4 HemiAccum;
#ifdef AMBIENTOCCLUSION 
varying float AmbientOcclusion;
#endif
varying vec2 FogParams;

#ifdef SHADOWING
uniform mat4 gWorldShadowMatrix;
uniform mat4 gLightProjectedMatrix;
uniform vec4 gShadowXBiasYRange;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

#ifdef SHADOWING_NV
uniform vec4 gShadowPCFOffset0;
uniform vec4 gShadowPCFOffset1;
uniform vec4 gShadowPCFOffset2;
#endif

#ifdef TREE
uniform mat4 gWorldMatrix;
uniform vec4 gTreeWindHack;
#endif

void main()
{
#ifdef TREE
   #define INPUT_POS gl_Vertex
   #define INPUT_NORMAL gl_Normal
   
	// tree deformation
	vec4 r9;
    r9.x = INPUT_POS.x - gTreeWindHack.y + gWorldMatrix[0][3];
    r9.y = INPUT_POS.y;
    r9.z = INPUT_POS.z - gTreeWindHack.z + gWorldMatrix[2][3];
    r9.w = INPUT_POS.w;

    float r1x =  gTreeWindHack.x * max(r9.x, r9.z);
	r1x *= 2.0;				// KLC -- Make tree swaying a little more obvious
    float r0y = sin(r1x);
	float r0x = cos(r1x);

	vec4 r5;
    r5.x = r0x;
    r5.y = 0.0;
    r5.z = -r0y;
    r5.w = gTreeWindHack.y - gWorldMatrix[0][3];

	vec4 r7;
    r7.x = r0y;
    r7.y = 0.0;
    r7.z = r0x;
    r7.w = gTreeWindHack.z - gWorldMatrix[2][3];

	vec4 r0;
    r0.x = dot(r9, r5);
    r0.y = gl_Vertex.y;
    r0.z = dot(r9, r7);
    r0.w = 1.0;
    
	vec3 r1;
    r1.x = dot(INPUT_NORMAL.xyz, r5.xyz);
    r1.y = INPUT_NORMAL.y;
    r1.z = dot(INPUT_NORMAL.xyz, r7.xyz);
   
   // Outputs are in r0/r1
   #define POS r0
   #define NORMAL r1
#else
   // Not tree-deformed so just use unmodified pos/normal.
   #define POS gl_Vertex
   #define NORMAL gl_Normal
#endif

   // Transform position
   gl_Position = POS * gWorldViewProjMatrix;
   
   // Set color = diffuse * gSunColor
   DirAccum.rgb = max(0.0, dot(NORMAL.xyz, gSunDir.xyz)) * gPreMultSun.rgb;
   DirAccum.w   = 1.0;
      
   HemiAccum.rgb = mix( gHemiBottomColor.rgb, gHemiTopColor.rgb, dot(NORMAL, gHemiAxis.xyz) * 0.5 + 0.5 ) * gHemiIntensity;
   HemiAccum.w   = 0.0;


#ifdef VERTEXCOLOR
   // Modulate with vertex color
   OutDirAccum.rgb  = OutDirAccum.rgb * SRGBToLinear(gl_Color);
   OutHemiAccum.rgb = OutHemiAccum.rgb * SRGBToLinear(gl_Color);
#endif


   // Set tex coords
#ifdef CLIFF
   gl_TexCoord[0].st = gl_MultiTexCoord2.st;
#else
   gl_TexCoord[0].st = gl_MultiTexCoord0.st;
#endif

	// Compute shadow coords
	
#if defined(SHADOWING) && defined(SHADOWING_NV)
	vec4 coords = gl_Vertex * gWorldShadowMatrix;
	coords.y = 1.0 - coords.y;  // Invert y (reading from a GL buffer)
	ShadowMapCoords0 = coords + gShadowPCFOffset0;
	ShadowMapCoords1 = coords + gShadowPCFOffset1;
	ShadowMapCoords2 = coords + gShadowPCFOffset2;
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ShadowMapCoords0 = gl_Vertex * gWorldShadowMatrix;

	vec4 lightSpacePosition = gl_Vertex * gLightProjectedMatrix;
	ShadowMapCoords1.x = lightSpacePosition.z * gShadowXBiasYRange.x;
#endif


#ifdef AMBIENTOCCLUSION   
   //-- pass through the ambientocclusion term
   AmbientOcclusion = gl_Color.a;
#endif

  
	// Compute Fog

	vec4 viewSpacePos = POS * gWorldViewMatrix;		// view space (no projection)
	
	FogParams.x = ComputeSquaredExponentialFogDensity( gFogDensity * max(0.0, viewSpacePos.z-gFogStart) / 1000.0);

	float perpDist = dot(viewSpacePos.xyz, gPlanarFogEquation.xyz);
	perpDist -= gPlanarFogEquation.w;

	float t;
	if( perpDist > 0.0 )
		t = 0.0;
	else
	{
		vec3  rayDirection = normalize(-viewSpacePos.xyz);
		float dirAlong     = dot(gPlanarFogEquation.xyz, rayDirection);
		float posAlong     = dot(gPlanarFogEquation.xyz, viewSpacePos.xyz);

		if (abs(dirAlong) < 0.00125)
			dirAlong = 1000.0;
		else
			dirAlong = -1.0 / dirAlong;

		t = gPlanarFogEquation.w - posAlong;
		t *= dirAlong;
	}

	FogParams.y = ComputeSquaredExponentialFogDensity( gPlanarFogDensity * t * 0.01 );   

#undef INPUT_NORMAL
#undef POS
#undef NORMAL
}


//---------------------
// FUNCTION_VS FlatVS

uniform mat4  gWorldViewProjMatrix;
uniform mat4  gWorldViewMatrix;
uniform vec4  gPreMultSun;
uniform vec4  gAmbientColorLinear;
uniform float gFogDensity;
uniform float gFogStart;
uniform float gPlanarFogDensity;
uniform vec4  gPlanarFogEquation;

varying vec4 DirAccum;
varying vec4 HemiAccum;
varying vec2 FogParams;

#ifdef SHADOWING
uniform mat4 gWorldShadowMatrix;
uniform mat4 gLightProjectedMatrix;
uniform vec4 gShadowXBiasYRange;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

#ifdef SHADOWING_NV
uniform vec4 gShadowPCFOffset0;
uniform vec4 gShadowPCFOffset1;
uniform vec4 gShadowPCFOffset2;
#endif

void main()      
{
   // Transform position
   gl_Position = gl_Vertex * gWorldViewProjMatrix;

   gl_FrontColor.rgb = gPreMultSun.rgb;
   
#ifdef VERTEXCOLOR
   // Modulate with vertex color
   gl_FrontColor.rgb *= SRGBToLinear( gl_Color ).rgb;
#endif

   // We can't take the normal into account here, so just apply ambient lighting.
   
   DirAccum.rgb *= gAmbientColorLinear.rgb;	// KLC, don't we usually add ambient?
   DirAccum.a = 1.0;
   
   HemiAccum = DirAccum;
   
   // Set tex coords
   gl_TexCoord[0].st = gl_MultiTexCoord0.st;

	// Compute shadow coords
	
#if defined(SHADOWING) && defined(SHADOWING_NV)
	vec4 coords = gl_Vertex * gWorldShadowMatrix;
	coords.y = 1.0 - coords.y;  // Invert y (reading from a GL buffer)
	ShadowMapCoords0 = coords + gShadowPCFOffset0;
	ShadowMapCoords1 = coords + gShadowPCFOffset1;
	ShadowMapCoords2 = coords + gShadowPCFOffset2;
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ShadowMapCoords0 = gl_Vertex * gWorldShadowMatrix;

	vec4 lightSpacePosition = gl_Vertex * gLightProjectedMatrix;
	ShadowMapCoords1.x = lightSpacePosition.z * gShadowXBiasYRange.x;
#endif

	// Compute Fog

	vec4 viewSpacePos = gl_Vertex * gWorldViewMatrix;		// view space (no projection)
	
	FogParams.x = ComputeSquaredExponentialFogDensity( gFogDensity * max(0.0, viewSpacePos.z-gFogStart) / 1000.0);

	float perpDist = dot(viewSpacePos.xyz, gPlanarFogEquation.xyz);
	perpDist -= gPlanarFogEquation.w;

	float t;
	if( perpDist > 0.0 )
		t = 0.0;
	else
	{
		vec3  rayDirection = normalize(-viewSpacePos.xyz);
		float dirAlong     = dot(gPlanarFogEquation.xyz, rayDirection);
		float posAlong     = dot(gPlanarFogEquation.xyz, viewSpacePos.xyz);

		if (abs(dirAlong) < 0.00125)
			dirAlong = 1000.0;
		else
			dirAlong = -1.0 / dirAlong;

		t = gPlanarFogEquation.w - posAlong;
		t *= dirAlong;
	}

	FogParams.y = ComputeSquaredExponentialFogDensity( gPlanarFogDensity * t * 0.01 );   
}   


//------------------------
// FUNCTION_VS BumpVS

uniform mat4  gWorldViewProjMatrix;
uniform mat4  gWorldViewMatrix;
uniform vec4  gModelSpaceCameraPos;
uniform vec4  gSunDir;
uniform vec4  gHemiAxis;
uniform float gFogDensity;
uniform float gFogStart;
uniform float gPlanarFogDensity;
uniform vec4  gPlanarFogEquation;

#ifndef TERRAIN
attribute vec3 Tangent;
#endif
#ifdef AMBIENTOCCLUSION 
varying float AmbientOcclusion;
#endif
varying vec3 SunDir;
varying vec3 HalfVec;
varying vec3 HemiAxis;
varying vec2 FogParams;

#ifdef SHADOWING
uniform mat4 gWorldShadowMatrix;
uniform mat4 gLightProjectedMatrix;
uniform vec4 gShadowXBiasYRange;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

#ifdef SHADOWING_NV
uniform vec4 gShadowPCFOffset0;
uniform vec4 gShadowPCFOffset1;
uniform vec4 gShadowPCFOffset2;
#endif

void main()
{
	// Transform position
	gl_Position = gl_Vertex * gWorldViewProjMatrix;

	// Compute binormal.
#ifndef TERRAIN
	// If tangent is halved in length, that means the binormal is flipped, so we
	// construct a negated binormal and rescale the tangent by 2.0.
	float tanLenSqr = dot(Tangent, Tangent);
	vec3 binormal;
	vec3 vTangent = Tangent;
	if( tanLenSqr < 0.3 )
	{
		// Rescale tangent.
		vTangent *= 2.0;
		
		// Flipped binormal.
		binormal = cross(-vTangent, gl_Normal);
	}
	else
	{
		// Standard binormal.
		binormal = -cross(-vTangent, gl_Normal);
	}
	vTangent = -vTangent;
#else   
//???	vec3 binormal = normalize(gl_Normal.xzy * vec3(0.0, 1.0, -1.0));
	vec3 binormal = gl_Normal.xzy * vec3(0.0, 1.0, -1.0);
   
	//-- compute tangent
	vec3 temp2 = gl_Normal.zxy * binormal.yzx;
//???	vec3 vTangent = normalize( (((InNormal.yzx) * (-binormal.zxy)) - temp2) );
	vec3 vTangent = (gl_Normal.yzx * -binormal.zxy) - temp2;
#endif
      
	// Move sun direction into tangent space.
	SunDir.x = dot(vTangent, gSunDir.xyz);
	SunDir.y = dot(binormal, gSunDir.xyz);
	SunDir.z = dot(gl_Normal.xyz, gSunDir.xyz);
	
	// Move Hemi Axis into tangent space
	HemiAxis.x = dot(vTangent, gHemiAxis.xyz);
	HemiAxis.y = dot(binormal, gHemiAxis.xyz);
	HemiAxis.z = dot(gl_Normal, gHemiAxis.xyz);      

	// Compute H vector.
	vec3 modelSpaceDirToCamera = normalize(gModelSpaceCameraPos.xyz - gl_Vertex.xyz);
	vec3 halfAngleVector = normalize(gSunDir.xyz + modelSpaceDirToCamera);
   
	// Move H vector into tangent space.
	HalfVec.x = dot(vTangent, halfAngleVector);
	HalfVec.y = dot(binormal, halfAngleVector);
	HalfVec.z = dot(gl_Normal, halfAngleVector);
//???   HalfVec = normalize(halfVec);

	// Set tex coords.  Diffuse and bump map share same sampling.
#ifdef CLIFF
    gl_TexCoord[0].st = gl_MultiTexCoord2.st;
#else
    gl_TexCoord[0].st = gl_MultiTexCoord0.st;
#endif

	// Compute shadow coords
	
#if defined(SHADOWING) && defined(SHADOWING_NV)
	vec4 coords = gl_Vertex * gWorldShadowMatrix;
	coords.y = 1.0 - coords.y;  // Invert y (reading from a GL buffer)
	ShadowMapCoords0 = coords + gShadowPCFOffset0;
	ShadowMapCoords1 = coords + gShadowPCFOffset1;
	ShadowMapCoords2 = coords + gShadowPCFOffset2;
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ShadowMapCoords0 = gl_Vertex * gWorldShadowMatrix;

	vec4 lightSpacePosition = gl_Vertex * gLightProjectedMatrix;
	ShadowMapCoords1.x = lightSpacePosition.z * gShadowXBiasYRange.x;
#endif

#ifdef AMBIENTOCCLUSION   
   //-- pass through the ambientocclusion term
   AmbientOcclusion = gl_Color.a;
#endif

#ifdef VERTEXCOLOR
   gl_FrontColor = gl_Color;
#endif 

	// Compute Fog

	vec4 viewSpacePos = gl_Vertex * gWorldViewMatrix;		// view space (no projection)
	
	FogParams.x = ComputeSquaredExponentialFogDensity( gFogDensity * max(0.0, viewSpacePos.z-gFogStart) / 1000.0);

	float perpDist = dot(viewSpacePos.xyz, gPlanarFogEquation.xyz);
	perpDist -= gPlanarFogEquation.w;

	float t;
	if( perpDist > 0.0 )
		t = 0.0;
	else
	{
		vec3  rayDirection = normalize(-viewSpacePos.xyz);
		float dirAlong     = dot(gPlanarFogEquation.xyz, rayDirection);
		float posAlong     = dot(gPlanarFogEquation.xyz, viewSpacePos.xyz);

		if (abs(dirAlong) < 0.00125)
			dirAlong = 1000.0;
		else
			dirAlong = -1.0 / dirAlong;

		t = gPlanarFogEquation.w - posAlong;
		t *= dirAlong;
	}

	FogParams.y = ComputeSquaredExponentialFogDensity( gPlanarFogDensity * t * 0.01 );   
}


//------------------------
// FUNCTION_PS DefaultPS

uniform sampler2D gSampler0;	// diffuse texture
uniform sampler2D gSampler4;	// terrain mask
#ifdef SHADOWING
#ifdef SHADOWING_NV
uniform sampler2DShadow gSampler7;
#endif
#ifndef SHADOWING_NV
uniform sampler2D gSampler7;
#endif
#endif

#ifdef TINT
uniform vec4 gTintColorLinear;
#endif
#ifdef OVERALLALPHA
uniform vec4 gFlatColor;
#endif
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif
#ifdef TREE
uniform vec4 gTreeColorVariation;
#endif
#ifdef BLENDCOLOR
uniform vec4 gBlendColor;
#endif
uniform vec4 gPlanarFogColor;
uniform vec4 gFogColor;
#ifdef SHADOWING
uniform vec4 gShadowColor;
#endif

varying vec4 DirAccum;
varying vec4 HemiAccum;
varying vec2 FogParams;
#ifdef AMBIENTOCCLUSION 
varying float AmbientOcclusion;
#endif
#ifdef SHADOWING
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

void main()
{   
	vec4 textureSample = texture2D(gSampler0, gl_TexCoord[0].st);
	textureSample.rgb *= textureSample.rgb;

	vec4 ret = textureSample * DirAccum;    

#ifdef BLENDCOLOR
	ret.rgb *= gBlendColor.rgb;
#endif

#ifdef TREE
	ret.rgb += gTreeColorVariation.rgb;
#endif

#if defined(SHADOWING) && defined(SHADOWING_NV)
	ret.rgb *= getShadowValueNV(gSampler7, ShadowMapCoords0, ShadowMapCoords1, ShadowMapCoords2, gShadowColor.x);
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ret.rgb *= getShadowingValue(gSampler7, ShadowMapCoords0, ShadowMapCoords1.x, gShadowColor.x);
#endif

#ifdef TINT
	ret.rgb += gTintColorLinear.rgb;
#endif

#ifdef AMBIENTOCCLUSION
	ret.rgb += AmbientOcclusion * HemiAccum.rgb * textureSample.rgb;
#endif
#if( !defined(AMBIENTOCCLUSION) && defined(BLENDCOLOR) )
	ret.rgb += HemiAccum.rgb * textureSample.rgb * gBlendColor.rgb;
#endif
#if( !defined(AMBIENTOCCLUSION) && !defined(BLENDCOLOR) )
	ret.rgb += HemiAccum.rgb * textureSample.rgb;
#endif    

	ret.rgb = mix( gPlanarFogColor.rgb, ret.rgb, FogParams.y );
	ret.rgb = mix( gFogColor.rgb, ret.rgb, FogParams.x );

#ifdef TERRAINBLEND
	// Sample terrain mask texture and multiply
	vec4 terrainMaskSample = texture2D(gSampler4, gl_TexCoord[0].st);
	ret.a = 1.0 - terrainMaskSample.a;
#endif

#ifdef OVERALLALPHA
	ret.a *= gFlatColor.a;
#endif

#ifdef SCALERGBOUTPUT
	ret.rgb *= gRGBScale;
#endif

	gl_FragColor = ret;
}


//------------------------------
// FUNCTION_VS DefaultColoredVS

// this is confusingly named, but is for pure, unlit color passthrough

uniform mat4  gWorldViewProjMatrix;

void main()
{
	// Transform position
	gl_Position = gl_Vertex * gWorldViewProjMatrix;
   
	// Linearize color.
	gl_FrontColor = SRGBToLinear( gl_Color );
   
	// Pass through tex coord.
	gl_TexCoord[0].st = gl_MultiTexCoord0.st;
}


//---------------------
// FUNCTION_PS BumpPS

uniform sampler2D gSampler0;	// diffuse texture
uniform sampler2D gSampler1;	// normal map
#ifdef TERRAINBLEND
uniform sampler2D gSampler4;
#endif

#ifdef PIXELXFORM
uniform vec4 gPixelXFormColor;
#endif
#ifdef TERRAIN
uniform float gTerrainBumpScale;
uniform float gTerrainMetalness;
uniform float gPreMultSunTerrainSpec;
uniform float gTerrainSpecularPower;
#endif
#ifndef TERRAIN
uniform float gBumpScale;
uniform float gSceneMetalness;
uniform float gPreMultSunSceneSpec;
uniform float gSceneSpecularPower;
#endif
uniform vec4 gPreMultSun;
uniform vec4 gSunColorLinear;
uniform vec4 gHemiBottomColor;
uniform vec4 gHemiTopColor;
uniform float gHemiIntensity;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif
#ifdef TINT
uniform vec4 gTintColorLinear;
#endif
uniform vec4 gFlatColor;
uniform vec4 gPlanarFogColor;
uniform vec4 gFogColor;

varying vec3 SunDir;
varying vec3 HalfVec;
varying vec3 HemiAxis;
varying vec2 FogParams;
#ifdef AMBIENTOCCLUSION 
varying float AmbientOcclusion;
#endif

#ifdef SHADOWING
#ifdef SHADOWING_NV
uniform sampler2DShadow gSampler7;
#endif
#ifndef SHADOWING_NV
uniform sampler2D gSampler7;
#endif
#endif

#ifdef SHADOWING
uniform vec4 gShadowColor;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

void main()
{
#ifndef TERRAIN   
	vec3 nHalfVec = normalize(HalfVec);
#endif

	vec4 diffSample = texture2D(gSampler0, gl_TexCoord[0].st);

//#ifdef ALPHATEST   
//	if( diffSample.a < 0.5 )
//		discard;
//#endif

#ifdef PIXELXFORM
	diffSample.rgb *= mix(gPixelXFormColor.rgb, vec3(1.0, 1.0, 1.0), diffSample.a);
#endif   
	diffSample.rgb *= diffSample.rgb;
   
	// Rescale to -1...1 range and swap a/r channels.
	vec4 temp = texture2D(gSampler1, gl_TexCoord[0].st);
	float normalpha = temp.r;
	vec3 norm = (temp.agb - 0.5) * 2.0;
   
#ifdef TERRAIN
	norm.x *= gTerrainBumpScale;
	norm.y *= gTerrainBumpScale;
#else
	norm.x *= gBumpScale;
	norm.y *= gBumpScale;
#endif
	norm = normalize(norm);   
   
	vec4 ret;
	vec3 spec = vec3(0.0, 0.0, 0.0);

	float nDotL = max(0.0, dot(norm, SunDir));   

	ret.rgb = gPreMultSun.rgb * diffSample.rgb;
   
#if( defined(SPEC) && defined(TERRAIN) && defined(SPECMASK) )
	vec3 specColor = mix( diffSample.rgb, gSunColorLinear.rgb, gTerrainMetalness );
	spec = pow( clamp(dot(norm, nHalfVec), 0.0, 1.0), gTerrainSpecularPower ) * gPreMultSunTerrainSpec * specColor;
	spec *= normalpha;
    ret.rgb += spec;
#endif

#if( defined(SPEC) && defined(TERRAIN) && !defined(SPECMASK) )
	vec3 specColor = mix( diffSample.rgb, gSunColorLinear.rgb, gTerrainMetalness );
	spec = pow( clamp(dot(norm, nHalfVec), 0.0, 1.0), gTerrainSpecularPower ) * gPreMultSunTerrainSpec * specColor;
    ret.rgb += spec;
#endif

#if( defined(SPEC) && !defined(TERRAIN) && defined(SPECMASK) )
	vec3 specColor = mix( diffSample.rgb, gSunColorLinear.rgb, gSceneMetalness );
	spec = pow( clamp(dot(norm, nHalfVec), 0.0, 1.0), gSceneSpecularPower ) * gPreMultSunSceneSpec * specColor;
	spec *= normalpha;
    ret.rgb += spec;
#endif

#if( defined(SPEC) && !defined(TERRAIN) && !defined(SPECMASK) )
	vec3 specColor = mix( diffSample.rgb, gSunColorLinear.rgb, gSceneMetalness );
	spec = pow( clamp(dot(norm, nHalfVec), 0.0, 1.0), gSceneSpecularPower ) * gPreMultSunSceneSpec * specColor;
    ret.rgb += spec;
#endif
   
	ret.rgb *= nDotL;

#if defined(SHADOWING) && defined(SHADOWING_NV)
	ret.rgb *= getShadowValueNV(gSampler7, ShadowMapCoords0, ShadowMapCoords1, ShadowMapCoords2, gShadowColor.x);
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV) && !defined(TERRAIN)
	ret.rgb *= getShadowingValue(gSampler7, ShadowMapCoords0, ShadowMapCoords1.x, gShadowColor.x);
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV) && defined(TERRAIN)
	ret.rgb *= getShadowingValue(gSampler7, ShadowMapCoords0, ShadowMapCoords1.x - 0.00005, gShadowColor.x);
#endif

#ifdef TINT
	ret.rgb += gTintColorLinear.rgb;
#endif

	vec3 hemiAccum = mix( gHemiBottomColor.rgb, gHemiTopColor.rgb, dot(norm, HemiAxis) * 0.5 + 0.5) * gHemiIntensity;
#ifdef AMBIENTOCCLUSION   
	ret.rgb += AmbientOcclusion * hemiAccum * diffSample.rgb;                     
#else
	ret.rgb += hemiAccum * diffSample.rgb;
#endif   
      
#ifdef ALPHATEST   
	ret.a = diffSample.a;
#else
	ret.a = 1.0;
#endif

#ifdef TERRAINBLEND
	// Sample terrain mask texture and multiply
	vec4 terrainMaskSample = texture2D(gSampler4, gl_TexCoord[0].st);
	ret.a = 1.0 - terrainMaskSample.a;
#endif

#ifdef TERRAINDIFFUSE
	// Use alpha from bump texture (times vertex alpha)
	ret.a = normalpha * gl_Color.a;
#endif

#ifdef BUMPDECAL	
	// Use alpha from diffuse texture (times flat color alpha)
	ret.a = diffSample.a * gFlatColor.a * gl_Color.a;
#endif   

	ret.rgb = mix( gPlanarFogColor.rgb, ret.rgb, FogParams.y );
	ret.rgb = mix( gFogColor.rgb, ret.rgb, FogParams.x );
     
#ifdef OVERALLALPHA
	ret.a *= gFlatColor.a;
#endif

#ifdef SCALERGBOUTPUT
	ret.rgb *= gRGBScale;
#endif

	gl_FragColor = ret;
}


//--------------------------
// FUNCTION_PS PixelXFormPS

uniform sampler2D gSampler0;	// diffuse texture
uniform sampler2D gSampler1;	// normal map
#ifdef SHADOWING
#ifdef SHADOWING_NV
uniform sampler2DShadow gSampler7;
#endif
#ifndef SHADOWING_NV
uniform sampler2D gSampler7;
#endif
#endif

#ifdef DECALPIXELXFORM
uniform vec4 gPixelXFormColorLinear;
#endif
#ifndef DECALPIXELXFORM
uniform vec4 gPixelXFormColor;
#endif
#ifdef TINT
uniform vec4 gTintColorLinear;
#endif
#ifdef OVERALLALPHA
uniform vec4 gFlatColor;
#endif
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif
uniform vec4 gPlanarFogColor;
uniform vec4 gFogColor;
#ifdef SHADOWING
uniform vec4 gShadowColor;
#endif

varying vec4 DirAccum;
varying vec4 HemiAccum;
varying vec2 FogParams;
#ifdef SHADOWING
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

void main()
{
	//base texture
	vec4 baseSample = texture2D(gSampler0, gl_TexCoord[0].st); 
	vec4 outColor;

#ifndef DECALPIXELXFORM
	// Get either white or pixelxform color based on diffuse texture's alpha channel.
	baseSample.rgb *= mix( gPixelXFormColor.rgb, vec3(1.0,1.0,1.0), baseSample.a );
	outColor = DirAccum;
#else
	outColor = DirAccum * gPixelXFormColorLinear;
#endif
	baseSample.rgb *= baseSample.rgb;

#ifdef TINT
   outColor = baseSample * outColor + gTintColorLinear;
#else
   outColor = baseSample * outColor;
#endif

#if defined(SHADOWING) && defined(SHADOWING_NV)
	outColor.rgb *= getShadowValueNV(gSampler7, ShadowMapCoords0, ShadowMapCoords1, ShadowMapCoords2, gShadowColor.x);
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	outColor.rgb *= getShadowingValue(gSampler7, ShadowMapCoords0, ShadowMapCoords1.x, gShadowColor.x);
#endif

	//-- apply hemi lighting
	outColor.rgb += HemiAccum.rgb * baseSample.rgb;

	outColor.rgb = mix( gPlanarFogColor.rgb, outColor.rgb, FogParams.y );
	outColor.rgb = mix( gFogColor.rgb, outColor.rgb, FogParams.x );

	outColor.a = 1.0;

#ifdef SCALERGBOUTPUT
	outColor.rgb *= gRGBScale;
#endif
#ifdef OVERALLALPHA
	outColor.a *= gFlatColor.a;
#endif

	gl_FragColor = outColor;
}


//------------------------
// FUNCTION_VS SparkleVS

uniform mat4  gWorldViewProjMatrix;
uniform vec4  gSunDir;

const vec4 gUVScalar = vec4(1.0f, -1.0f, 0.0f, 0.0f);

void main()
{
   gl_Position = gl_Vertex * gWorldViewProjMatrix;
   gl_TexCoord[0].st = gl_MultiTexCoord0.st + gSparkleOffset.xy;
   gl_TexCoord[1].st = (gl_MultiTexCoord0.st * gUVScalar.xy) - gSparkleOffset.xy;
   gl_TexCoord[2].st = gl_MultiTexCoord0.st;

   gl_FrontColor.rgb = dot(gl_Normal.rgb, gSunDir.rgb);
   gl_FrontColor.a = 0.0; 
}


//------------------------
// FUNCTION_PS SparklePS

uniform sampler2D gSampler0;
uniform sampler2D gSampler1;
uniform sampler2D gSampler4;

#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif

void main()
{
	vec4 textureSample0 = texture2D(gSampler0, gl_TexCoord[0].st);
	textureSample0.rgb *= textureSample0.rgb;
	vec4 textureSample1 = texture2D(gSampler1, gl_TexCoord[1].st);
	textureSample1.rgb *= textureSample1.rgb;
   
	vec4 ret = textureSample0 * textureSample1 * (1.0 - gl_Color);

#ifdef TERRAINBLEND
	// Sample terrain mask texture and multiply
	vec4 terrainMaskSample = texture2D(gSampler4, gl_TexCoord[2].st);
	ret.a *= 1.0 - terrainMaskSample.a;
#endif

#ifdef SCALERGBOUTPUT
	ret.rgb *= gRGBScale;
#endif

	gl_FragColor = ret;
}


//----------------------
// FUNCTION_VS DecalVS

uniform mat4  gWorldViewProjMatrix;
uniform mat4  gWorldViewMatrix;
uniform vec4  gSunDir;
uniform vec4  gPreMultSun;
uniform vec4  gHemiAxis;
uniform vec4  gHemiTopColor;
uniform vec4  gHemiBottomColor;
uniform float gHemiIntensity;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif
uniform float gFogDensity;
uniform float gFogStart;
uniform float gPlanarFogDensity;
uniform vec4  gPlanarFogEquation;

varying vec4 DirAccum;
varying vec4 HemiAccum;
varying vec2 FogParams;

#ifdef SHADOWING
uniform mat4 gWorldShadowMatrix;
uniform mat4 gLightProjectedMatrix;
uniform vec4 gShadowXBiasYRange;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

#ifdef SHADOWING_NV
uniform vec4 gShadowPCFOffset0;
uniform vec4 gShadowPCFOffset1;
uniform vec4 gShadowPCFOffset2;
#endif

void main()
{
	// Transform position
	gl_Position = gl_Vertex * gWorldViewProjMatrix;

	// Set color = diffuse * gSunColorr
	DirAccum.rgb = max( 0.0, dot(gl_Normal.xyz, gSunDir.xyz)) * gPreMultSun.rgb;
	DirAccum.w   = 1.0;

	HemiAccum.rgb = mix( gHemiBottomColor.rgb, gHemiTopColor.rgb, dot(gl_Normal, gHemiAxis.xyz) * 0.5 + 0.5) * gHemiIntensity;
	HemiAccum.a   = 0.0;

	gl_TexCoord[0].st = gl_MultiTexCoord0.st;
	gl_FrontColor = gl_Color;

	// Compute shadow coords
	
#if defined(SHADOWING) && defined(SHADOWING_NV)
	vec4 coords = gl_Vertex * gWorldShadowMatrix;
	coords.y = 1.0 - coords.y;  // Invert y (reading from a GL buffer)
	ShadowMapCoords0 = coords + gShadowPCFOffset0;
	ShadowMapCoords1 = coords + gShadowPCFOffset1;
	ShadowMapCoords2 = coords + gShadowPCFOffset2;
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ShadowMapCoords0 = gl_Vertex * gWorldShadowMatrix;

	vec4 lightSpacePosition = gl_Vertex * gLightProjectedMatrix;
	ShadowMapCoords1.x = lightSpacePosition.z * gShadowXBiasYRange.x;
#endif

	// Compute Fog

	vec4 viewSpacePos = gl_Vertex * gWorldViewMatrix;		// view space (no projection)
	
	FogParams.x = ComputeSquaredExponentialFogDensity( gFogDensity * max(0.0, viewSpacePos.z-gFogStart) / 1000.0);

	float perpDist = dot(viewSpacePos.xyz, gPlanarFogEquation.xyz);
	perpDist -= gPlanarFogEquation.w;

	float t;
	if( perpDist > 0.0 )
		t = 0.0;
	else
	{
		vec3  rayDirection = normalize(-viewSpacePos.xyz);
		float dirAlong     = dot(gPlanarFogEquation.xyz, rayDirection);
		float posAlong     = dot(gPlanarFogEquation.xyz, viewSpacePos.xyz);

		if (abs(dirAlong) < 0.00125)
			dirAlong = 1000.0;
		else
			dirAlong = -1.0 / dirAlong;

		t = gPlanarFogEquation.w - posAlong;
		t *= dirAlong;
	}

	FogParams.y = ComputeSquaredExponentialFogDensity( gPlanarFogDensity * t * 0.01 );   
}

//----------------------
// FUNCTION_PS DecalPS

uniform sampler2D gSampler0;
#ifdef SHADOWING
#ifdef SHADOWING_NV
uniform sampler2DShadow gSampler7;
#endif
#ifndef SHADOWING_NV
uniform sampler2D gSampler7;
#endif
#endif

uniform vec4 gFlatColor;
uniform vec4 gPlanarFogColor;
uniform vec4 gFogColor;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif
#ifdef SHADOWING
uniform vec4 gShadowColor;
#endif

varying vec4 DirAccum;
varying vec4 HemiAccum;
varying vec2 FogParams;
#ifdef SHADOWING
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

void main()
{
   vec4 textureSample0 = SRGBToLinear( texture2D(gSampler0, gl_TexCoord[0].st) );
      
   vec4 ret;   
   //-- modulate 2X color
   ret.rgb = textureSample0.rgb * gl_Color.rgb * 2.0;
   //-- modulate alpha
   ret.a  = textureSample0.a * gFlatColor.a * gl_Color.a;

   //-- modulate directional light accumulation
   ret.rgb *= DirAccum.rgb;    

#if defined(SHADOWING) && defined(SHADOWING_NV)
	ret.rgb *= getShadowValueNV(gSampler7, ShadowMapCoords0, ShadowMapCoords1, ShadowMapCoords2, gShadowColor.x);
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ret.rgb *= getShadowingValue(gSampler7, ShadowMapCoords0, ShadowMapCoords1.x, gShadowColor.x);
#endif

   //-- add hemi lighting
   ret.rgb += HemiAccum.rgb * textureSample0.rgb;

   ret.rgb = mix( gPlanarFogColor.rgb, ret.rgb, FogParams.y );
   ret.rgb = mix( gFogColor.rgb, ret.rgb, FogParams.x );

#ifdef SCALERGBOUTPUT
   ret.rgb *= gRGBScale;
#endif   

   gl_FragColor = ret;
}


/*
float4 gFabricProperties
<
   string paramClass="param";
   string paramName="properties";
> = float4(0,0,0,0);

void FabricVS(
             in float4 InPosition  : POSITION,
             in float4 InNormal    : NORMAL,
             in float2 InTexCoord0 : TEXCOORD0,

             out float4 OutPosition   : POSITION,
             out float2 OutTexCoord0  : TEXCOORD0,
             out float4 OutDirAccum   : TEXCOORD1,   // directional diffuse
             out float4 OutHemiAccum  : TEXCOORD2,    // hemi accum
             out float4 OutNormal     : TEXCOORD3,            
             out float4 OutPos        : TEXCOORD4      

#ifdef SHADOWING
            ,out float4 OutShadowMapCoords0 : TEXCOORD5   // vertex texture coord 5 -> shadowmap texcoord
            ,out float4 OutShadowMapCoords1 : TEXCOORD6   // vertex texture coord 6 -> .x will hold the depth of the pixel in light-space
            ,out float4 OutShadowMapCoords2 : TEXCOORD8
#endif
             )
{
   // Transform position
   OutPosition = mul( InPosition, gWorldViewProjMatrix );

   // Set color = diffuse * gSunColorr
   OutDirAccum.rgb = max(0.0f, dot(InNormal.xyz, gSunDir.xyz)) * gPreMultSun;
   OutDirAccum.w   = 1.0f;

   //OutHemiAccum.rgb = lerp(gHemiBottomColor, gHemiTopColor, dot3(InNormal, gHemiAxis) * 0.5f + 0.5f) * gHemiIntensity;
   OutHemiAccum.rgb = ComputeHemiLight(InNormal, gHemiAxis);
   OutHemiAccum.w   = 0.0f;
   
   OutTexCoord0 = InTexCoord0;
   OutNormal    = InNormal;
   OutPos       = InPosition;

#ifdef SHADOWING
   ComputeShadowCoords(InPosition, OutShadowMapCoords0, OutShadowMapCoords1, OutShadowMapCoords2);   
#endif

}

float4 FabricPS(
               in float2 InTexCoord0 : TEXCOORD0,
               in float4 InDirAccum  : TEXCOORD1,
               in float4 InHemiAccum : TEXCOORD2,
               in float4 InNormal    : TEXCOORD3,
               in float4 InPosition  : TEXCOORD4

#ifdef SHADOWING
              ,in float4 InShadowMapCoords0 : TEXCOORD5
              ,in float4 InShadowMapCoords1 : TEXCOORD6
              ,in float4 InShadowMapCoords2 : TEXCOORD8
#endif
               ) : COLOR0
{
   float3 viewVec  = normalize(gModelSpaceCameraPos.xyz - InPosition.xyz);
   float cosView   = saturate(dot(viewVec.xyz, InNormal.xyz));
   
   //-- BTK -- code to compute the shine value the same way as in the ATI 9800 path
   float shine   = (pow(1.0f-(cosView*cosView), gFabricProperties.x) * gFabricProperties.y);
   
   float4 textureSample0 = SRGBToLinear(tex2D(gSampler0, InTexCoord0));
   
   float4 ret = textureSample0;
   //-- add in our specular shine
   ret.rgb += shine;

   //-- modulate directional light accumulation
   ret.rgb *= InDirAccum;    

#ifdef SHADOWING
   #ifdef SHADOWING_NV
      ret.rgb *= getShadowValueNV(InShadowMapCoords0, InShadowMapCoords1, InShadowMapCoords2);
   #else
      ret.rgb *= getShadowingValue(InShadowMapCoords0, InShadowMapCoords1.x);
   #endif
#endif

   //-- add hemi lighting
   ret.rgb += InHemiAccum * textureSample0;

#ifdef SCALERGBOUTPUT
   ret.rgb *= gRGBScale;
#endif

#ifdef OVERALLALPHA
   ret.a *= gFlatColor.a;
#endif

   return ret;
}
*/


//--------------------------------
// FUNCTION_VS DefaultFlatColorVS

uniform mat4  gWorldViewProjMatrix;

void main()
{
   // Transform position
   gl_Position = gl_Vertex * gWorldViewProjMatrix;
}


//--------------------------------
// FUNCTION_PS DefaultFlatColorPS

uniform vec4 gFlatColorLinear;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif

void main()
{
   vec4 ret = gFlatColorLinear;
#ifdef SCALERGBOUTPUT
   ret.rgb *= gRGBScale;
#endif
   gl_FragColor = ret;
}


//----------------------------------
// FUNCTION_VS AlphatestFlatColorVS

uniform mat4  gWorldViewProjMatrix;

void main()
{
   // Transform position
   gl_Position = gl_Vertex * gWorldViewProjMatrix;
   gl_TexCoord[0].st = gl_MultiTexCoord0.st;
}


//----------------------------------
// FUNCTION_PS AlphatestFlatColorPS

uniform sampler2D gSampler0;
uniform vec4 gFlatColorLinear;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif

void main()
{
   vec4 ret = gFlatColorLinear;
   ret.a *= texture2D(gSampler0, gl_TexCoord[0].st).a;
#ifdef SCALERGBOUTPUT
   ret.rgb *= gRGBScale;
#endif
   gl_FragColor = ret;
}


/*
void TreeShadowGenVS( 
   in  float4 InPosition  : POSITION

#ifdef ALPHATEST
  ,in  float2 InTexCoord0 : TEXCOORD0
#endif

  ,out float4 OutPosition   : POSITION   // vertex position 
  ,out float2 OutTexCoord0  : TEXCOORD0  // vertex texture coord 0
#ifdef ALPHATEST
  ,out float2 OutTexCoord1  : TEXCOORD1   // vertex texture coord 1
#endif
  )
{
   #define INPUT_POS InPosition
   // Deform.
   #include "tree_deform.inc"
   // Outputs are in r0/r1
   #define POS r0

   // Transform position
   OutPosition = mul( POS, gWorldViewProjMatrix );

   // Output depth   
   OutTexCoord0.x = (OutPosition.z * gShadowXBiasYRange.x) + gShadowXBiasYRange.y;
   OutTexCoord0.y = 0.0f;   

   #ifdef ALPHATEST
      OutTexCoord1 = InTexCoord0;
   #endif

#undef INPUT_POS   
#undef POS
}

void TreeShadowMapVS(
   in  float4 InPosition  : POSITION
  ,in  float4 InNormal    : NORMAL
  ,in  float4 InTexCoord0 : TEXCOORD0
  ,out float4 OutPosition : POSITION
  ,out float4 OutColor0   : COLOR0
  ,out float4 OutTexCoord0: TEXCOORD0
  ,out float4 OutTexCoord1: TEXCOORD1
  ,out float4 OutTexCoord2: TEXCOORD2
  )
{
   #define INPUT_POS InPosition
   #define INPUT_NORMAL InNormal
   // Deform.
   #include "tree_deform.inc"
   // Outputs are in r0/r1
   #define POS r0
   #define NORMAL r1

   // Transform position
   OutPosition = mul( POS, gWorldViewProjMatrix );

   float NormalDotLight = max(dot3(NORMAL.xyz, gSunDir), 0);
   OutColor0.w = 1.0f;
   OutColor0.rgb = min(gSunColor.rgb * NormalDotLight, 1);

   OutTexCoord0 = mul(InPosition, gWorldShadowMatrix);
   OutTexCoord1 = InTexCoord0;

   float4 temp = mul(InPosition, gLightProjectedMatrix);
   OutTexCoord2 = float4(temp.z * gShadowXBiasYRange.x, 0, 0, 0);   

#undef POS
#undef INPUT_NORMAL
#undef NORMAL
}

void TreePostSceneShadowingVS(
   in  float4 InPosition  : POSITION
#ifdef ALPHATEST
  ,in  float4 InTexCoord0 : TEXCOORD0
#endif
  ,out float4 OutPosition : POSITION
  ,out float4 OutColor0   : COLOR0
  ,out float4 OutTexCoord0: TEXCOORD0
  ,out float4 OutTexCoord1: TEXCOORD1
#ifdef ALPHATEST
  ,out float4 OutTexCoord2: TEXCOORD2
#endif
   )
{
   #define INPUT_POS InPosition
   // Deform.
   #include "tree_deform.inc"
   // Outputs are in r0/r1
   #define POS r0 

   // Transform position
   OutPosition = mul( POS, gWorldViewProjMatrix );
   OutColor0 = float4(0,0,0,0);
   OutTexCoord0 = mul(POS, gWorldShadowMatrix);

   //-- Depth in light space
   float4 depthInLightSpace = mul(POS, gLightProjectedMatrix);
#ifdef ALPHATEST
   OutTexCoord1 = InTexCoord0;
   OutTexCoord2 = float4(depthInLightSpace.z * gShadowXBiasYRange.x, 0, 0,0);
#else
   OutTexCoord1 = float4(depthInLightSpace.z * gShadowXBiasYRange.x, 0, 0,0);
#endif
}

void TreePostSceneShadowing_NV_VS(
   in  float4 InPosition  : POSITION
#ifdef ALPHATEST
  ,in  float4 InTexCoord0 : TEXCOORD0
#endif
  ,out float4 OutPosition : POSITION
  ,out float4 OutColor0   : COLOR0
  ,out float2 OutTexCoord0: TEXCOORD0
#ifdef ALPHATEST
  ,out float2 OutTexCoord1: TEXCOORD1
#endif
  ,out float2 OutTexCoord2: TEXCOORD2
  ,out float2 OutTexCoord3: TEXCOORD3
   )
{
   #define INPUT_POS InPosition
   // Deform.
   #include "tree_deform.inc"
   // Outputs are in r0/r1
   #define POS r0 

   // Transform position
   OutPosition = mul( POS, gWorldViewProjMatrix );
   //-- output black coloer
   OutColor0 = float4(0,0,0,0);

   //-- transform position
   r0 = mul(POS, gWorldShadowMatrix);

   //-- PCF Filter
   OutTexCoord0 = r0 + gShadowPCFOffset0;
   OutTexCoord2 = r0 + gShadowPCFOffset1;
   OutTexCoord3 = r0 + gShadowPCFOffset2;

#ifdef ALPHATEST
   OutTexCoord1 = InTexCoord0;
#endif 
}

float4 TreePostSceneShadowing_NV_PS(
    in float2 InTexCoord0 : TEXCOORD0
   ,in float2 InTexCoord1 : TEXCOORD1
   ,in float2 InTexCoord2 : TEXCOORD2
   ,in float2 InTexCoord3 : TEXCOORD3) : COLOR0
{
#ifdef ALPHATEST
   float4 textureSample1 = tex2D(gSampler1, InTexCoord1);
#endif

   float4 textureSample0  = tex2D(gSampler0, InTexCoord0) * 0.333333f;
   textureSample0        += tex2D(gSampler0, InTexCoord2) * 0.333333f;
   textureSample0        += tex2D(gSampler0, InTexCoord3) * 0.333333f;
   
   float4 outColor = saturate(textureSample0 + gShadowColor);
   outColor        = saturate(outColor + gAmbientColor);
   
#ifdef ALPHATEST
   //-- this is some bogus ported code
   float alpha = 1.0f - textureSample1;
   outColor = saturate(outColor + alpha);
#endif
   return outColor;
}

void TreeShadowGen_NV_VS( 
   in  float4 InPosition  : POSITION
  ,in  float2 InTexCoord0 : TEXCOORD0
  ,out float4 OutPosition : POSITION   // vertex position 
  ,out float2 OutTexCoord0: TEXCOORD0
  )
{
   #define INPUT_POS InPosition
   // Deform.
   #include "tree_deform.inc"
   // Outputs are in r0/r1
   #define POS r0
   // Transform position
   OutPosition = mul( POS, gWorldViewProjMatrix );
   OutTexCoord0 = InTexCoord0;

#undef INPUT_POS   
#undef POS
}


float4 TreeShadowGen_NV_PS(in float2 InTexCoord0 : TEXCOORD0) : COLOR0
{
   float4 sample = tex2D(gSampler0, InTexCoord0);   
   return sample;
}
*/

//------------------------------
// FUNCTION_VS DecalSelectionVS

uniform mat4  gWorldViewProjMatrix;

#ifdef SHADOWING
uniform mat4 gWorldShadowMatrix;
uniform mat4 gLightProjectedMatrix;
uniform vec4 gShadowXBiasYRange;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

#ifdef SHADOWING_NV
uniform vec4 gShadowPCFOffset0;
uniform vec4 gShadowPCFOffset1;
uniform vec4 gShadowPCFOffset2;
#endif

void main()
{
   // Transform position
   gl_Position = gl_Vertex * gWorldViewProjMatrix;
   gl_TexCoord[0].st = gl_MultiTexCoord0.st;
   gl_FrontColor = gl_Color;

	// Compute shadow coords
	
#if defined(SHADOWING) && defined(SHADOWING_NV)
	vec4 coords = gl_Vertex * gWorldShadowMatrix;
	coords.y = 1.0 - coords.y;  // Invert y (reading from a GL buffer)
	ShadowMapCoords0 = coords + gShadowPCFOffset0;
	ShadowMapCoords1 = coords + gShadowPCFOffset1;
	ShadowMapCoords2 = coords + gShadowPCFOffset2;
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	ShadowMapCoords0 = gl_Vertex * gWorldShadowMatrix;

	vec4 lightSpacePosition = gl_Vertex * gLightProjectedMatrix;
	ShadowMapCoords1.x = lightSpacePosition.z * gShadowXBiasYRange.x;
#endif
}


//------------------------------
// FUNCTION_PS DecalSelectionPS

uniform sampler2D gSampler0;
uniform vec4 gFlatColorLinear;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif

#ifdef SHADOWING
#ifdef SHADOWING_NV
uniform sampler2DShadow gSampler7;
#endif
#ifndef SHADOWING_NV
uniform sampler2D gSampler7;
#endif
#endif

#ifdef SHADOWING
uniform vec4 gShadowColor;
varying vec4 ShadowMapCoords0;
varying vec4 ShadowMapCoords1;
varying vec4 ShadowMapCoords2;
#endif

void main()
{
   vec4 diffuse = SRGBToLinear( texture2D(gSampler0, gl_TexCoord[0].st) );

   vec4 outColor = diffuse * gFlatColorLinear;

#if defined(SHADOWING) && defined(SHADOWING_NV)
	outColor.rgb *= getShadowValueNV(gSampler7, ShadowMapCoords0, ShadowMapCoords1, ShadowMapCoords2, gShadowColor.x);
#endif

#if defined(SHADOWING) && !defined(SHADOWING_NV)
	outColor.rgb *= getShadowingValue(gSampler7, ShadowMapCoords0, ShadowMapCoords1.x, gShadowColor.x);
#endif

#ifdef SCALERGBOUTPUT
   outColor.rgb *= gRGBScale;
#endif

   gl_FragColor = outColor;
}


//------------------------------
// FUNCTION_PS DefaultColoredPS

uniform sampler2D gSampler0;
#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif

void main()
{
   vec4 diffuse = SRGBToLinear( texture2D(gSampler0, gl_TexCoord[0].st) );
   vec4 outColor = diffuse * gl_Color;

#ifdef SCALERGBOUTPUT
   outColor.rgb *= gRGBScale;
#endif

   gl_FragColor = outColor;
}
